"use strict";
/********************************************************************************
 * Copyright (C) 2018 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.connectInlineDebugAdapter = exports.connectPipeDebugAdapter = exports.connectSocketDebugAdapter = exports.startDebugAdapter = void 0;
const net = require("net");
const child_process_1 = require("child_process");
const inline_debug_adapter_1 = require("@theia/debug/lib/node/inline-debug-adapter");
const stream_debug_adapter_1 = require("@theia/debug/lib/node/stream-debug-adapter");
const isElectron = require('is-electron');
/**
 * Starts debug adapter process.
 */
function startDebugAdapter(executable) {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const options = { stdio: ['pipe', 'pipe', 2] };
    if (executable.options) {
        options.cwd = executable.options.cwd;
        // The additional environment of the executed program or shell. If omitted
        // the parent process' environment is used. If provided it is merged with
        // the parent process' environment.
        options.env = Object.assign({}, process.env);
        Object.assign(options.env, executable.options.env);
    }
    let childProcess;
    const { command, args } = executable;
    if (command === 'node') {
        if (Array.isArray(args) && args.length > 0) {
            const forkOptions = {
                env: options.env,
                // When running in Electron, fork will automatically add ELECTRON_RUN_AS_NODE=1 to the env,
                // but this will cause issues when debugging Electron apps, so we'll remove it.
                execArgv: isElectron()
                    ? ['-e', 'delete process.env.ELECTRON_RUN_AS_NODE;require(process.argv[1])']
                    : [],
                silent: true
            };
            if (options.cwd) {
                forkOptions.cwd = options.cwd;
            }
            options.stdio.push('ipc');
            forkOptions.stdio = options.stdio;
            childProcess = child_process_1.fork(args[0], args.slice(1), forkOptions);
        }
        else {
            throw new Error(`It is not possible to launch debug adapter with the command: ${JSON.stringify(executable)}`);
        }
    }
    else {
        childProcess = child_process_1.spawn(command, args, options);
    }
    return new stream_debug_adapter_1.ProcessDebugAdapter(childProcess);
}
exports.startDebugAdapter = startDebugAdapter;
/**
 * Connects to a remote debug server.
 */
function connectSocketDebugAdapter(server) {
    const socket = net.createConnection(server.port, server.host);
    return new stream_debug_adapter_1.SocketDebugAdapter(socket);
}
exports.connectSocketDebugAdapter = connectSocketDebugAdapter;
function connectPipeDebugAdapter(adapter) {
    const socket = net.createConnection(adapter.path);
    return new stream_debug_adapter_1.SocketDebugAdapter(socket);
}
exports.connectPipeDebugAdapter = connectPipeDebugAdapter;
function connectInlineDebugAdapter(adapter) {
    return new inline_debug_adapter_1.InlineDebugAdapter(adapter.implementation);
}
exports.connectInlineDebugAdapter = connectInlineDebugAdapter;
//# sourceMappingURL=plugin-debug-adapter-starter.js.map