/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
import { Emitter } from '../../../base/common/event';
import { FoldingRegions } from './foldingRanges';
var FoldingModel = /** @class */ (function () {
    function FoldingModel(textModel, decorationProvider) {
        this._updateEventEmitter = new Emitter();
        this.onDidChange = this._updateEventEmitter.event;
        this._textModel = textModel;
        this._decorationProvider = decorationProvider;
        this._regions = new FoldingRegions(new Uint32Array(0), new Uint32Array(0));
        this._editorDecorationIds = [];
        this._isInitialized = false;
    }
    Object.defineProperty(FoldingModel.prototype, "regions", {
        get: function () { return this._regions; },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(FoldingModel.prototype, "textModel", {
        get: function () { return this._textModel; },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(FoldingModel.prototype, "isInitialized", {
        get: function () { return this._isInitialized; },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(FoldingModel.prototype, "decorationProvider", {
        get: function () { return this._decorationProvider; },
        enumerable: false,
        configurable: true
    });
    FoldingModel.prototype.toggleCollapseState = function (toggledRegions) {
        var _this = this;
        if (!toggledRegions.length) {
            return;
        }
        toggledRegions = toggledRegions.sort(function (r1, r2) { return r1.regionIndex - r2.regionIndex; });
        var processed = {};
        this._decorationProvider.changeDecorations(function (accessor) {
            var e_1, _a;
            var k = 0; // index from [0 ... this.regions.length]
            var dirtyRegionEndLine = -1; // end of the range where decorations need to be updated
            var lastHiddenLine = -1; // the end of the last hidden lines
            var updateDecorationsUntil = function (index) {
                while (k < index) {
                    var endLineNumber = _this._regions.getEndLineNumber(k);
                    var isCollapsed = _this._regions.isCollapsed(k);
                    if (endLineNumber <= dirtyRegionEndLine) {
                        accessor.changeDecorationOptions(_this._editorDecorationIds[k], _this._decorationProvider.getDecorationOption(isCollapsed, endLineNumber <= lastHiddenLine));
                    }
                    if (isCollapsed && endLineNumber > lastHiddenLine) {
                        lastHiddenLine = endLineNumber;
                    }
                    k++;
                }
            };
            try {
                for (var toggledRegions_1 = __values(toggledRegions), toggledRegions_1_1 = toggledRegions_1.next(); !toggledRegions_1_1.done; toggledRegions_1_1 = toggledRegions_1.next()) {
                    var region = toggledRegions_1_1.value;
                    var index = region.regionIndex;
                    var editorDecorationId = _this._editorDecorationIds[index];
                    if (editorDecorationId && !processed[editorDecorationId]) {
                        processed[editorDecorationId] = true;
                        updateDecorationsUntil(index); // update all decorations up to current index using the old dirtyRegionEndLine
                        var newCollapseState = !_this._regions.isCollapsed(index);
                        _this._regions.setCollapsed(index, newCollapseState);
                        dirtyRegionEndLine = Math.max(dirtyRegionEndLine, _this._regions.getEndLineNumber(index));
                    }
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (toggledRegions_1_1 && !toggledRegions_1_1.done && (_a = toggledRegions_1.return)) _a.call(toggledRegions_1);
                }
                finally { if (e_1) throw e_1.error; }
            }
            updateDecorationsUntil(_this._regions.length);
        });
        this._updateEventEmitter.fire({ model: this, collapseStateChanged: toggledRegions });
    };
    FoldingModel.prototype.update = function (newRegions, blockedLineNumers) {
        var _this = this;
        if (blockedLineNumers === void 0) { blockedLineNumers = []; }
        var newEditorDecorations = [];
        var isBlocked = function (startLineNumber, endLineNumber) {
            var e_2, _a;
            try {
                for (var blockedLineNumers_1 = __values(blockedLineNumers), blockedLineNumers_1_1 = blockedLineNumers_1.next(); !blockedLineNumers_1_1.done; blockedLineNumers_1_1 = blockedLineNumers_1.next()) {
                    var blockedLineNumber = blockedLineNumers_1_1.value;
                    if (startLineNumber < blockedLineNumber && blockedLineNumber <= endLineNumber) { // first line is visible
                        return true;
                    }
                }
            }
            catch (e_2_1) { e_2 = { error: e_2_1 }; }
            finally {
                try {
                    if (blockedLineNumers_1_1 && !blockedLineNumers_1_1.done && (_a = blockedLineNumers_1.return)) _a.call(blockedLineNumers_1);
                }
                finally { if (e_2) throw e_2.error; }
            }
            return false;
        };
        var lastHiddenLine = -1;
        var initRange = function (index, isCollapsed) {
            var startLineNumber = newRegions.getStartLineNumber(index);
            var endLineNumber = newRegions.getEndLineNumber(index);
            if (isCollapsed && isBlocked(startLineNumber, endLineNumber)) {
                isCollapsed = false;
            }
            newRegions.setCollapsed(index, isCollapsed);
            var maxColumn = _this._textModel.getLineMaxColumn(startLineNumber);
            var decorationRange = {
                startLineNumber: startLineNumber,
                startColumn: Math.max(maxColumn - 1, 1),
                endLineNumber: startLineNumber,
                endColumn: maxColumn
            };
            newEditorDecorations.push({ range: decorationRange, options: _this._decorationProvider.getDecorationOption(isCollapsed, endLineNumber <= lastHiddenLine) });
            if (isCollapsed && endLineNumber > lastHiddenLine) {
                lastHiddenLine = endLineNumber;
            }
        };
        var i = 0;
        var nextCollapsed = function () {
            while (i < _this._regions.length) {
                var isCollapsed = _this._regions.isCollapsed(i);
                i++;
                if (isCollapsed) {
                    return i - 1;
                }
            }
            return -1;
        };
        var k = 0;
        var collapsedIndex = nextCollapsed();
        while (collapsedIndex !== -1 && k < newRegions.length) {
            // get the latest range
            var decRange = this._textModel.getDecorationRange(this._editorDecorationIds[collapsedIndex]);
            if (decRange) {
                var collapsedStartLineNumber = decRange.startLineNumber;
                if (decRange.startColumn === Math.max(decRange.endColumn - 1, 1) && this._textModel.getLineMaxColumn(collapsedStartLineNumber) === decRange.endColumn) { // test that the decoration is still covering the full line else it got deleted
                    while (k < newRegions.length) {
                        var startLineNumber = newRegions.getStartLineNumber(k);
                        if (collapsedStartLineNumber >= startLineNumber) {
                            initRange(k, collapsedStartLineNumber === startLineNumber);
                            k++;
                        }
                        else {
                            break;
                        }
                    }
                }
            }
            collapsedIndex = nextCollapsed();
        }
        while (k < newRegions.length) {
            initRange(k, false);
            k++;
        }
        this._editorDecorationIds = this._decorationProvider.deltaDecorations(this._editorDecorationIds, newEditorDecorations);
        this._regions = newRegions;
        this._isInitialized = true;
        this._updateEventEmitter.fire({ model: this });
    };
    /**
     * Collapse state memento, for persistence only
     */
    FoldingModel.prototype.getMemento = function () {
        var collapsedRanges = [];
        for (var i = 0; i < this._regions.length; i++) {
            if (this._regions.isCollapsed(i)) {
                var range = this._textModel.getDecorationRange(this._editorDecorationIds[i]);
                if (range) {
                    var startLineNumber = range.startLineNumber;
                    var endLineNumber = range.endLineNumber + this._regions.getEndLineNumber(i) - this._regions.getStartLineNumber(i);
                    collapsedRanges.push({ startLineNumber: startLineNumber, endLineNumber: endLineNumber });
                }
            }
        }
        if (collapsedRanges.length > 0) {
            return collapsedRanges;
        }
        return undefined;
    };
    /**
     * Apply persisted state, for persistence only
     */
    FoldingModel.prototype.applyMemento = function (state) {
        var e_3, _a;
        if (!Array.isArray(state)) {
            return;
        }
        var toToogle = [];
        try {
            for (var state_1 = __values(state), state_1_1 = state_1.next(); !state_1_1.done; state_1_1 = state_1.next()) {
                var range = state_1_1.value;
                var region = this.getRegionAtLine(range.startLineNumber);
                if (region && !region.isCollapsed) {
                    toToogle.push(region);
                }
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (state_1_1 && !state_1_1.done && (_a = state_1.return)) _a.call(state_1);
            }
            finally { if (e_3) throw e_3.error; }
        }
        this.toggleCollapseState(toToogle);
    };
    FoldingModel.prototype.dispose = function () {
        this._decorationProvider.deltaDecorations(this._editorDecorationIds, []);
    };
    FoldingModel.prototype.getAllRegionsAtLine = function (lineNumber, filter) {
        var result = [];
        if (this._regions) {
            var index = this._regions.findRange(lineNumber);
            var level = 1;
            while (index >= 0) {
                var current = this._regions.toRegion(index);
                if (!filter || filter(current, level)) {
                    result.push(current);
                }
                level++;
                index = current.parentIndex;
            }
        }
        return result;
    };
    FoldingModel.prototype.getRegionAtLine = function (lineNumber) {
        if (this._regions) {
            var index = this._regions.findRange(lineNumber);
            if (index >= 0) {
                return this._regions.toRegion(index);
            }
        }
        return null;
    };
    FoldingModel.prototype.getRegionsInside = function (region, filter) {
        var result = [];
        var index = region ? region.regionIndex + 1 : 0;
        var endLineNumber = region ? region.endLineNumber : Number.MAX_VALUE;
        if (filter && filter.length === 2) {
            var levelStack = [];
            for (var i = index, len = this._regions.length; i < len; i++) {
                var current = this._regions.toRegion(i);
                if (this._regions.getStartLineNumber(i) < endLineNumber) {
                    while (levelStack.length > 0 && !current.containedBy(levelStack[levelStack.length - 1])) {
                        levelStack.pop();
                    }
                    levelStack.push(current);
                    if (filter(current, levelStack.length)) {
                        result.push(current);
                    }
                }
                else {
                    break;
                }
            }
        }
        else {
            for (var i = index, len = this._regions.length; i < len; i++) {
                var current = this._regions.toRegion(i);
                if (this._regions.getStartLineNumber(i) < endLineNumber) {
                    if (!filter || filter(current)) {
                        result.push(current);
                    }
                }
                else {
                    break;
                }
            }
        }
        return result;
    };
    return FoldingModel;
}());
export { FoldingModel };
/**
 * Collapse or expand the regions at the given locations
 * @param levels The number of levels. Use 1 to only impact the regions at the location, use Number.MAX_VALUE for all levels.
 * @param lineNumbers the location of the regions to collapse or expand, or if not set, all regions in the model.
 */
export function toggleCollapseState(foldingModel, levels, lineNumbers) {
    var e_4, _a;
    var toToggle = [];
    var _loop_1 = function (lineNumber) {
        var region = foldingModel.getRegionAtLine(lineNumber);
        if (region) {
            var doCollapse_1 = !region.isCollapsed;
            toToggle.push(region);
            if (levels > 1) {
                var regionsInside = foldingModel.getRegionsInside(region, function (r, level) { return r.isCollapsed !== doCollapse_1 && level < levels; });
                toToggle.push.apply(toToggle, __spreadArray([], __read(regionsInside)));
            }
        }
    };
    try {
        for (var lineNumbers_1 = __values(lineNumbers), lineNumbers_1_1 = lineNumbers_1.next(); !lineNumbers_1_1.done; lineNumbers_1_1 = lineNumbers_1.next()) {
            var lineNumber = lineNumbers_1_1.value;
            _loop_1(lineNumber);
        }
    }
    catch (e_4_1) { e_4 = { error: e_4_1 }; }
    finally {
        try {
            if (lineNumbers_1_1 && !lineNumbers_1_1.done && (_a = lineNumbers_1.return)) _a.call(lineNumbers_1);
        }
        finally { if (e_4) throw e_4.error; }
    }
    foldingModel.toggleCollapseState(toToggle);
}
/**
 * Collapse or expand the regions at the given locations including all children.
 * @param doCollapse Whether to collapse or expand
 * @param levels The number of levels. Use 1 to only impact the regions at the location, use Number.MAX_VALUE for all levels.
 * @param lineNumbers the location of the regions to collapse or expand, or if not set, all regions in the model.
 */
export function setCollapseStateLevelsDown(foldingModel, doCollapse, levels, lineNumbers) {
    var e_5, _a;
    if (levels === void 0) { levels = Number.MAX_VALUE; }
    var toToggle = [];
    if (lineNumbers && lineNumbers.length > 0) {
        try {
            for (var lineNumbers_2 = __values(lineNumbers), lineNumbers_2_1 = lineNumbers_2.next(); !lineNumbers_2_1.done; lineNumbers_2_1 = lineNumbers_2.next()) {
                var lineNumber = lineNumbers_2_1.value;
                var region = foldingModel.getRegionAtLine(lineNumber);
                if (region) {
                    if (region.isCollapsed !== doCollapse) {
                        toToggle.push(region);
                    }
                    if (levels > 1) {
                        var regionsInside = foldingModel.getRegionsInside(region, function (r, level) { return r.isCollapsed !== doCollapse && level < levels; });
                        toToggle.push.apply(toToggle, __spreadArray([], __read(regionsInside)));
                    }
                }
            }
        }
        catch (e_5_1) { e_5 = { error: e_5_1 }; }
        finally {
            try {
                if (lineNumbers_2_1 && !lineNumbers_2_1.done && (_a = lineNumbers_2.return)) _a.call(lineNumbers_2);
            }
            finally { if (e_5) throw e_5.error; }
        }
    }
    else {
        var regionsInside = foldingModel.getRegionsInside(null, function (r, level) { return r.isCollapsed !== doCollapse && level < levels; });
        toToggle.push.apply(toToggle, __spreadArray([], __read(regionsInside)));
    }
    foldingModel.toggleCollapseState(toToggle);
}
/**
 * Collapse or expand the regions at the given locations including all parents.
 * @param doCollapse Whether to collapse or expand
 * @param levels The number of levels. Use 1 to only impact the regions at the location, use Number.MAX_VALUE for all levels.
 * @param lineNumbers the location of the regions to collapse or expand.
 */
export function setCollapseStateLevelsUp(foldingModel, doCollapse, levels, lineNumbers) {
    var e_6, _a;
    var toToggle = [];
    try {
        for (var lineNumbers_3 = __values(lineNumbers), lineNumbers_3_1 = lineNumbers_3.next(); !lineNumbers_3_1.done; lineNumbers_3_1 = lineNumbers_3.next()) {
            var lineNumber = lineNumbers_3_1.value;
            var regions = foldingModel.getAllRegionsAtLine(lineNumber, function (region, level) { return region.isCollapsed !== doCollapse && level <= levels; });
            toToggle.push.apply(toToggle, __spreadArray([], __read(regions)));
        }
    }
    catch (e_6_1) { e_6 = { error: e_6_1 }; }
    finally {
        try {
            if (lineNumbers_3_1 && !lineNumbers_3_1.done && (_a = lineNumbers_3.return)) _a.call(lineNumbers_3);
        }
        finally { if (e_6) throw e_6.error; }
    }
    foldingModel.toggleCollapseState(toToggle);
}
/**
 * Collapse or expand a region at the given locations. If the inner most region is already collapsed/expanded, uses the first parent instead.
 * @param doCollapse Whether to collapse or expand
 * @param lineNumbers the location of the regions to collapse or expand.
 */
export function setCollapseStateUp(foldingModel, doCollapse, lineNumbers) {
    var e_7, _a;
    var toToggle = [];
    try {
        for (var lineNumbers_4 = __values(lineNumbers), lineNumbers_4_1 = lineNumbers_4.next(); !lineNumbers_4_1.done; lineNumbers_4_1 = lineNumbers_4.next()) {
            var lineNumber = lineNumbers_4_1.value;
            var regions = foldingModel.getAllRegionsAtLine(lineNumber, function (region) { return region.isCollapsed !== doCollapse; });
            if (regions.length > 0) {
                toToggle.push(regions[0]);
            }
        }
    }
    catch (e_7_1) { e_7 = { error: e_7_1 }; }
    finally {
        try {
            if (lineNumbers_4_1 && !lineNumbers_4_1.done && (_a = lineNumbers_4.return)) _a.call(lineNumbers_4);
        }
        finally { if (e_7) throw e_7.error; }
    }
    foldingModel.toggleCollapseState(toToggle);
}
/**
 * Folds or unfolds all regions that have a given level, except if they contain one of the blocked lines.
 * @param foldLevel level. Level == 1 is the top level
 * @param doCollapse Whether to collapse or expand
*/
export function setCollapseStateAtLevel(foldingModel, foldLevel, doCollapse, blockedLineNumbers) {
    var filter = function (region, level) { return level === foldLevel && region.isCollapsed !== doCollapse && !blockedLineNumbers.some(function (line) { return region.containsLine(line); }); };
    var toToggle = foldingModel.getRegionsInside(null, filter);
    foldingModel.toggleCollapseState(toToggle);
}
/**
 * Folds all regions for which the lines start with a given regex
 * @param foldingModel the folding model
 */
export function setCollapseStateForMatchingLines(foldingModel, regExp, doCollapse) {
    var editorModel = foldingModel.textModel;
    var regions = foldingModel.regions;
    var toToggle = [];
    for (var i = regions.length - 1; i >= 0; i--) {
        if (doCollapse !== regions.isCollapsed(i)) {
            var startLineNumber = regions.getStartLineNumber(i);
            if (regExp.test(editorModel.getLineContent(startLineNumber))) {
                toToggle.push(regions.toRegion(i));
            }
        }
    }
    foldingModel.toggleCollapseState(toToggle);
}
/**
 * Folds all regions of the given type
 * @param foldingModel the folding model
 */
export function setCollapseStateForType(foldingModel, type, doCollapse) {
    var regions = foldingModel.regions;
    var toToggle = [];
    for (var i = regions.length - 1; i >= 0; i--) {
        if (doCollapse !== regions.isCollapsed(i) && type === regions.getType(i)) {
            toToggle.push(regions.toRegion(i));
        }
    }
    foldingModel.toggleCollapseState(toToggle);
}
