/*******************************************************************************
 * Copyright (c) 2000, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 *******************************************************************************/
package org.eclipse.dltk.internal.ui.typehierarchy;

import org.eclipse.dltk.core.IMethod;
import org.eclipse.dltk.core.IType;
import org.eclipse.dltk.core.ITypeHierarchy;
import org.eclipse.dltk.core.ModelException;
import org.eclipse.dltk.core.ScriptModelUtil;
import org.eclipse.dltk.ui.ModelElementSorter;
import org.eclipse.dltk.ui.viewsupport.SourcePositionSorter;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;

public abstract class AbstractHierarchyViewerSorter extends ViewerComparator {

	private static final int OTHER = 10;
	private static final int CLASS = 20;
	private static final int CUMULATIVE_PART = 18;
	private static final int CUMULATIVE_CLASS = 19;
	private static final int ANONYM = 30;

	private ModelElementSorter fNormalSorter;
	private SourcePositionSorter fSourcePositonSorter;

	public AbstractHierarchyViewerSorter() {
		fNormalSorter = new ModelElementSorter();
		fSourcePositonSorter = new SourcePositionSorter();
	}

	protected abstract ITypeHierarchy getHierarchy(IType type);

	public abstract boolean isSortByDefiningType();

	public abstract boolean isSortAlphabetically();

	protected int getTypeFlags(IType type) throws ModelException {
		return type.getFlags();
	}

	@Override
	public int category(Object element) {
		if (element instanceof IType) {
			IType type = (IType) element;
			if (type.getElementName().length() == 0) {
				return ANONYM;
			}
			// try {
			// int flags = getTypeFlags(type);
			return CLASS;
			// } catch (ModelException e) {
			// ignore
			// }
		} else if (element instanceof CumulativeType) {
			return CUMULATIVE_CLASS;
		} else if (element instanceof CumulativeType.Part) {
			return CUMULATIVE_PART;
		}
		return OTHER;
	}

	@Override
	public int compare(Viewer viewer, Object e1, Object e2) {
		if (!isSortAlphabetically() && !isSortByDefiningType()) {
			return fSourcePositonSorter.compare(viewer, e1, e2);
		}

		int cat1 = category(e1);
		int cat2 = category(e2);

		if (cat1 != cat2)
			return cat1 - cat2;

		if (cat1 == OTHER) { // method or field
			if (isSortByDefiningType()) {
				try {
					IType def1 = (e1 instanceof IMethod)
							? getDefiningType((IMethod) e1)
							: null;
					IType def2 = (e2 instanceof IMethod)
							? getDefiningType((IMethod) e2)
							: null;
					if (def1 != null) {
						if (def2 != null) {
							if (!def2.equals(def1)) {
								return compareInHierarchy(def1, def2);
							}
						} else {
							return -1;
						}
					} else {
						if (def2 != null) {
							return 1;
						}
					}
				} catch (ModelException e) {

				}
			}
			if (isSortAlphabetically()) {
				return fNormalSorter.compare(viewer, e1, e2);
				// use appearance pref page settings
			}
			return 0;
		} else if (cat1 == ANONYM) {
			return 0;
		} else if (isSortAlphabetically()) {
			String name1 = getTypeName(e1);
			String name2 = getTypeName(e2);
			return getComparator().compare(name1, name2);
		}
		return 0;
	}

	private String getTypeName(Object e) {
		if (e instanceof IType) {
			return ((IType) e).getElementName();
		} else if (e instanceof CumulativeType) {
			return ((CumulativeType) e).getFirst().getElementName();
		} else if (e instanceof CumulativeType.Part) {
			return ((CumulativeType.Part) e).type.getSourceModule().getPath()
					.toString();
		} else {
			return e.toString();
		}
	}

	private IType getDefiningType(IMethod method) throws ModelException {
		// int flags= method.getFlags();
		// if (Flags.isPrivate(flags) || Flags.isStatic(flags) ||
		// method.isConstructor()) {
		// return null;
		// }
		//
		// IType declaringType= method.getDeclaringType();
		// MethodOverrideTester tester= new MethodOverrideTester(declaringType,
		// getHierarchy(declaringType));
		// IMethod res= tester.findDeclaringMethod(method, true);
		// if (res == null) {
		// return null;
		// }
		// return res.getDeclaringType();

		return null;
	}

	private int compareInHierarchy(IType def1, IType def2) {
		if (ScriptModelUtil.isSuperType(getHierarchy(def1), def2, def1)) {
			return 1;
		} else if (ScriptModelUtil.isSuperType(getHierarchy(def2), def1,
				def2)) {
			return -1;
		}

		String name1 = def1.getElementName();
		String name2 = def2.getElementName();

		return getComparator().compare(name1, name2);
	}

}
