/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.persistence.dom;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.ExternalFormHelper;
import org.eclipse.persistence.tools.mapping.persistence.ExternalPersistenceConfiguration;
import org.eclipse.persistence.tools.mapping.persistence.ExternalPersistenceUnit;
import org.eclipse.persistence.tools.mapping.persistence.PersistenceDocumentType;
import org.w3c.dom.Element;
import static org.eclipse.persistence.tools.mapping.persistence.PersistenceXmlConstants.*;

/**
 * The external form interacting with the XML document for the Persistence Configuration file.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public final class PersistenceConfiguration extends AbstractExternalForm
                                            implements ExternalPersistenceConfiguration {

	/**
	 * The helper is used when the document needs to be modified.
	 */
	private final ExternalFormHelper helper;

	/**
	 * Creates a new <code>PersistenceConfiguration</code>.
	 *
	 * @param helper The helper is used when the document needs to be modified
	 */
	public PersistenceConfiguration(ExternalFormHelper helper) {
		super(null);
		this.helper = helper;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalPersistenceUnit addPersistenceUnit(String name) {
		PersistenceUnit persistenceUnit = buildPersistenceUnit(-1);
		persistenceUnit.addSelf();
		persistenceUnit.setName(name);
		return persistenceUnit;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addSelf(String elementName) {
		return addSelf(elementName, Collections.<String>emptyList());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element addSelf(String elementName, List<String> elementNamesOrder) {
		return helper.buildPersistenceConfiguration(this);
	}

	private PersistenceUnit buildPersistenceUnit(int index) {
		return new PersistenceUnit(this, index);
	}

	@SuppressWarnings("incomplete-switch")
	public String buildSchemaLocation(PersistenceDocumentType type) {

		StringBuilder sb = new StringBuilder();
		sb.append(PERSISTENCE_NAMESPACE_URI);
		sb.append(" ");

		switch (type) {
			case JPA_1_0: sb.append(PERSISTENCE_XSD_URI_1_0); break;
			case JPA_2_0: sb.append(PERSISTENCE_XSD_URI_2_0); break;
			case JPA_2_1: sb.append(PERSISTENCE_XSD_URI_2_1); break;
		}

		return sb.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getBuildVersion() {
		return PersistenceDocumentType.JPA_2_1.getVersion();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public PersistenceDocumentType getDocumentType() {
		return PersistenceDocumentType.value(getNamespace(), getVersion(), getSchemaLocation());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Element getElement() {
		return getRootElement();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getElementName() {
		return PERSISTENCE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected ExternalFormHelper getHelper() {
		return helper;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public URL getLocation() {
		return helper.getLocation();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalPersistenceUnit getPersistenceUnit(int index) {

		if (hasChild(PersistenceUnit.PERSISTENCE_UNIT, index)) {
			return buildPersistenceUnit(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalPersistenceUnit> persistenceUnits() {

		int count = persistenceUnitsSize();
		List<ExternalPersistenceUnit> persistenceUnits = new ArrayList<ExternalPersistenceUnit>(count);

		for (int index = 0; index < count; index++) {
			persistenceUnits.add(buildPersistenceUnit(index));
		}

		return persistenceUnits;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int persistenceUnitsSize() {
		return getChildrenSize(PersistenceUnit.PERSISTENCE_UNIT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removePersistenceUnit(int index) {
		PersistenceUnit persistenceUnit = buildPersistenceUnit(index);
		persistenceUnit.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void save() throws Exception {
		helper.save();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDocumentType(PersistenceDocumentType type) {
		setAttribute(XSD_VERSION, type.getVersion());
		setSchemaLocation(buildSchemaLocation(type));
	}
}