/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.orm.ExternalManyToManyMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalMappingVisitor;
import org.eclipse.persistence.tools.mapping.orm.ExternalNoSqlJoinField;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form for a M:M mapping, which is a child of an entity.
 *
 * @see MappedSuperClass
 *
 * @version 2.6
 */
final class ManyToManyMapping extends ObjectCollectionMapping
                              implements ExternalManyToManyMapping {

	/**
	 * Creates a new <code>ManyToManyMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	ManyToManyMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void accept(ExternalMappingVisitor visitor) {
		visitor.visit(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalNoSqlJoinField addJoinField(String name) {
		NoSqlJoinField joinField = buildJoinField(-1);
		joinField.addSelf();
		joinField.setName(name);
		return joinField;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(TARGET_ENTITY);
		names.add(FETCH);
		names.add(ACCESS);
		names.add(MAPPED_BY);
		names.add(ATTRIBUTE_TYPE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(ORDER_BY);
		names.add(OrderColumn.ORDER_COLUMN);
		names.add(MAP_KEY);
		names.add(MAP_KEY_CLASS);
		names.add(MAP_KEY_TEMPORAL);
		names.add(MAP_KEY_ENUMERATED);
		names.add(MAP_KEY_CONVERT);
		names.add(MAP_KEY_ATTRIBUTE_OVERRIDE);
		names.add(MAP_KEY_ASSOCIATION_OVERRIDE);
		names.add(MAP_KEY_COLUMN);
		names.add(MAP_KEY_JOIN_COLUMN);
		names.add(JoinTable.JOIN_TABLE);
		names.add(CASCADE);
		names.add(JOIN_FETCH);
		names.add(BatchFetch.BATCH_FETCH);
		names.add(Property.PROPERTY);
		names.add(AccessMethods.ACCESS_METHODS);
		return names;
	}

	private NoSqlJoinField buildJoinField(int index) {
		return new NoSqlJoinField(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return MANY_TO_MANY;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMappedByMappingName() {
		return getAttribute(MAPPED_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMappedByMappingNameTextRange() {
		return getAttributeTextRange(MAPPED_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalNoSqlJoinField> joinFields() {

		int count = joinFieldSize();
		List<ExternalNoSqlJoinField> joinFields = new ArrayList<ExternalNoSqlJoinField>(count);

		for (int index = 0; index < count; index++) {
			joinFields.add(buildJoinField(index));
		}

		return joinFields;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int joinFieldSize() {
		return getChildrenSize(NoSqlJoinField.JOIN_FIELD);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeJoinField(int index) {
		removeChild(NoSqlJoinField.JOIN_FIELD, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMappedByMappingName(String name) {
		setAttribute(MAPPED_BY, name);
	}
}