/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm;

import java.util.List;
import org.eclipse.persistence.annotations.ExistenceType;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * This interface and inherited behavior describe the configurable properties defined for a
 * mapped superclass entity in the ORM xml. One or more mapped superclass entity may be declared
 * at the root of the ORM xml.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see javax.persistence.MappedSuperclass
 * @see http://jcp.org/aboutJava/communityprocess/final/jsr220/index.html, persistence specificaton
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public interface ExternalMappedSuperClass extends ExternalEmbeddable,
                                                  ExternalEntityListener,
                                                  ExternalEntityListenerHolder {

	/**
	 * The element name used to store and retrieve the cache-intercepter child node.
	 */
	String CACHE_INTERCEPTOR = "cache-interceptor";

	/**
	 * The attribute name used to store and retrieve the cacheable property
	 */
	String CACHEABLE = "cacheable";

	/**
	 * The element name used to store and retrieve the exclude-default-listeners child node.
	 */
	String EXCLUDE_DEFAULT_LISTENERS = "exclude-default-listeners";

	/**
	 * The element name used to store and retrieve the exclude-superclass-listeners child node.
	 */
	String EXCLUDE_SUPERCLASS_LISTENERS = "exclude-superclass-listeners";

	/**
	 * The attribute name used to store and retrieve the existence-checking property.
	 */
	String EXISTENCE_CHECKING = "existence-checking";

	/**
	 * The element name used to store and retrieve the id-class child node.
	 */
	String ID_CLASS = "id-class";

	/**
	 * The node name used to store and retrieve the element encapsulated by this external form.
	 */
	String MAPPED_SUPERCLASS = "mapped-superclass";

	/**
	 * The attribute name used to store and retrieve the method-name property of the callback child nodes.
	 */
	String METHOD_NAME = "method-name";

	/**
	 * The element name used to store and retrieve the multitenant child node.
	 */
	String MULTITENANT = "multitenant";

	/**
	 * The attribute name used to store and retrieve the optimistic-locking property.
	 */
	String OPTIMISTIC_LOCKING = "optimistic-locking";

	/**
	 * The element name used to store and retrieve the post-load child node.
	 */
	String POST_LOAD = "post-load";

	/**
	 * The element name used to store and retrieve the post-persist child node.
	 */
	String POST_PERSIST = "post-persist";

	/**
	 * The element name used to store and retrieve the post-remove child node.
	 */
	String POST_REMOVE = "post-remove";

	/**
	 * The element name used to store and retrieve the post-update child node.
	 */
	String POST_UPDATE = "post-update";

	/**
	 * The element name used to store and retrieve the pre-persist child node.
	 */
	String PRE_PERSIST = "pre-persist";

	/**
	 * The element name used to store and retrieve the pre-remove child node.
	 */
	String PRE_REMOVE = "pre-remove";

	/**
	 * The element name used to store and retrieve the pre-update child node.
	 */
	String PRE_UPDATE = "pre-update";

	/**
	 * The attribute name used to store and retrieve the primary-key property.
	 */
	String PRIMARY_KEY = "primary-key";

	/**
	 * The attribute name used to store and retrieve the read-only property.
	 */
	String READ_ONLY =	"read-only";

	/**
	 * Adds a cache to this entity.
	 */
	ExternalCache addCache();

	/**
	 * Adds an embedded id mapping with the given name.
	 */
	ExternalEmbeddedIDMapping addEmbeddedIdMapping(String name);

	/**
	 * Adds a fetch group with the given name.
	 */
	ExternalFetchGroup addFetchGroup(String name);

	/**
	 * Adds multitenancy to this entity.
	 */
	ExternalMultitenancyPolicy addMultitenancy();

	/**
	 * Adds a named query with the given name.
	 */
	ExternalNamedQuery addNamedQuery(String name);

	/**
	 * Adds a native query with the given name.
	 */
	ExternalNativeQuery addNativeQuery(String name);

	/**
	 * Create new Optimistic locking for this entity replacing the old one.
	 */
	ExternalOptimisticLocking addOptimisticLocking();

	/**
	 * Creates new primary key for this entity replacing the old one.
	 */
	ExternalPrimaryKey addPrimaryKey();

	/**
	 * Creates a new sequence generator for this entity.
	 */
	ExternalSequenceGenerator addSequenceGenerator();

	/**
	 * Adds a stored procedure query with the given name.
	 */
	ExternalNamedStoredProcedureQuery addStoredProcedureQuery(String name);

	/**
	 * Creates a new table generator for this entity.
	 */
	ExternalTableGenerator addTableGenerator();

	/**
	 * Adds a transformation mapping with the given name.
	 */
	ExternalTransformationMapping addTransformationMapping(String name);

	/**
	 * Adds a variable one-to-one mapping with the given name.
	 */
	ExternalVariableOneToOneMapping addVariableOneToOneMapping(String name);

	/**
	 * Adds a version mapping with the given name.
	 */
	ExternalVersionMapping addVersionMapping(String name);

	/**
	 * Returns a list of the fetch groups defined for this entity.
	 */
	List<ExternalFetchGroup> fetchGroups();

	/**
	 * Returns the count of fetch groups defined for this entity.
	 */
	int fetchGroupsSize();

	/**
	 * Returns the cache for this entity.
	 */
	ExternalCache getCache();

	/**
	 * Returns the {@link TextRange} of the cacheable property.
	 */
	TextRange getCacheableTextRange();

	/**
	 * Returns the cache intercepter class name defined for this entity.
	 */
	String getCacheInterceptorClassName();

	/**
	 * Returns the {@link TextRange} for the cache intercepter class name defined for this entity.
	 */
	TextRange getCacheInterceptorClassNameTextRange();

	/**
	 * Returns the {@link TextRange} of the excludes class listeners value.
	 */
	TextRange getExcludesDefaultListenersTextRange();

	/**
	 * Returns the {@link TextRange} for the excludes superclass listeners value.
	 */
	TextRange getExcludesSuperclassListenersTextRange();

	/**
	 * Returns the existence type for this entity
	 */
	ExistenceType getExistenceCheckingType();

	/**
	 * Returns the {@link TextRange} for the existence type for this entity
	 */
	TextRange getExistenceCheckingTypeTextRange();

	/**
	 * Returns the fetch group at the given position.
	 */
	ExternalFetchGroup getFetchGroup(int index);

	/**
	 * Returns the ID class name defined for this entity.
	 */
	String getIdClassName();

	/**
	 * Returns the {@link TextRange} for the ID class name defined for this entity.
	 */
	TextRange getIdClassNameTextRange();

	/**
	 * Returns the description of the mappings in this entity
	 */
	String getMappingsDescription();

	/**
	 * Returns the multitenancy policy for this entity.
	 */
	ExternalMultitenancyPolicy getMultitenancy();

	/**
	 * Returns the {@link TextRange} for the multitenant of this entity.
	 */
	TextRange getMultitenancyTextRange();

	/**
	 * Returns a named query with the given name.
	 */
	ExternalBasicNamedQuery getNamedQuery(int index);

	/**
	 * Returns a native query with the given name.
	 */
	ExternalNativeQuery getNativeQuery(int index);

	/**
	 * Returns the optimistic locking definition for this entity.
	 */
	ExternalOptimisticLocking getOptimisticLocking();

	/**
	 * Returns the primary key definition for this entity.
	 */
	ExternalPrimaryKey getPrimaryKey();

	/**
	 * Returns the {@link TextRange} of the read-only property.
	 */
	TextRange getReadOnlyTextRange();

	/**
	 * Returns the sequence generator defined for this entity if one exists.
	 */
	ExternalSequenceGenerator getSequenceGenerator();

	/**
	 * Returns the stored procedure query at the index.
	 */
	ExternalNamedStoredProcedureQuery getStoredProcedureQuery(int index);

	/**
	 * Returns the table generator for this entity if one exists.
	 */
	ExternalTableGenerator getTableGenerator();

	/**
	 * Determines whether the cache for this entity is defined or not.
	 */
	boolean hasCache();

	/**
	 * Determines whether the multitenancy policy for this entity is defined or not.
	 */
	boolean hasMultitenancy();

	/**
	 * Determines whether the optimistic locking definition for this entity is defined or not.
	 */
	boolean hasOptimisticLocking();

	/**
	 * Determines whether the primary key definition for this entity is defined or not.
	 */
	boolean hasPrimaryKey();

	/**
	 * Determines whether the sequence generator for this entity is defined or not.
	 */
	boolean hasSequenceGenerator();

	/**
	 * Determines whether the table generator for this entity is defined or not.
	 */
	boolean hasTableGenerator();

	/**
	 * Returns whether this entity is cacheable.
	 */
	Boolean isCacheable();

	/**
	 * Returns whether this entity is read only.
	 */
	Boolean isReadOnly();

	/**
	 * Returns a list of named queries defined for this entity.
	 */
	List<ExternalNamedQuery> namedQueries();

	/**
	 * Returns the count of named queried defined for this entity.
	 */
	int namedQueriesSize();

	/**
	 * Returns a list of native queries defined on this entity.
	 */
	List<ExternalNativeQuery> nativeQueries();

	/**
	 * Returns the count of native queries defined on this entity.
	 */
	int nativeQueriesSize();

	/**
	 * Removes the cache from this entity if it exists.
	 */
	void removeCache();

	/**
	 * Removes the fetch group specified at the given position.
	 */
	void removeFetchGroup(int index);

	/**
	 * Removes the multitenancy from this entity.
	 */
	void removeMultiTenancy();

	/**
	 * Removes the named query named.
	 */
	void removeNamedQuery(int index);

	/**
	 * Removes the native query named.
	 */
	void removeNativeQuery(int index);

	/**
	 * Removes the optimistic locking from this entity if it exists.
	 */
	void removeOptimisticLocking();

	/**
	 * Removes the primary Key definition from this entity if it exists.
	 */
	void removePrimaryKey();

	/**
	 * Removes the sequence generator from this entity.
	 */
	void removeSequenceGenerator();

	/**
	 * Removes the stored procedure query named.
	 */
	void removeStoredProcedureQuery(int index);

	/**
	 * Removes the table generator from this entity.
	 */
	void removeTableGenerator();

	/**
	 * Sets whether this entity is cacheable or not.
	 */
	void setCacheable(Boolean cacheable);

	/**
	 * Sets the name of the cache intercepter class declared for this entity.
	 */
	void setCacheInterceptorClassName(String className);

	/**
	 * Sets if this entity should exclude default class listener methods.
	 */
	void setExcludesDefaultListeners(boolean excludeDefaultListeners);

	/**
	 * Sets if this entity has declared to exclude listener methods defined on mapped superclasses.
	 */
	void setExcludesSuperclassListeners(boolean excludesSuperclassListeners);

	/**
	 * Sets the existence checking type for this entity.
	 */
	void setExistenceCheckingType(ExistenceType type);

	/**
	 * Sets the name of the ID class declared for this entity.
	 */
	void setIdClassName(String className);

	/**
	 * Sets the description for the mappings on this entity.
	 */
	void setMappingsDescription(String description);

	/**
	 * Sets whether this entity is read only.
	 */
	void setReadOnly(Boolean readOnly);

	/**
	 * Returns if this entity has declared to exclude default listener methods.
	 */
	boolean shouldExcludesDefaultListeners();

	/**
	 * Returns if this entity has declared to exclude listener methods defined on mapped superclasses.
	 */
	boolean shouldExcludesSuperclassListeners();

	/**
	 * Returns a list of named stored procedure queries defined for this entity.
	 */
	List<ExternalNamedStoredProcedureQuery> storedProcedureQueries();

	/**
	 * Returns the count of named stored procedure queried defined for this entity.
	 */
	int storedProcedureQueriesSize();
}