/*******************************************************************************
 * Copyright (c) 1998, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation from Oracle TopLink
******************************************************************************/
package org.eclipse.persistence.tools.db.relational.platformsmodel;

import org.eclipse.persistence.tools.utility.XMLTools;
import org.eclipse.persistence.tools.utility.node.AbstractNode;
import org.w3c.dom.Node;

/**
 * Common behavior(?).
 * Neither instance variable should ever be null.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
abstract class AbstractJDBCTypeToJavaTypeDeclarationMapping extends AbstractNode {

	JDBCType jdbcType;
	JavaTypeDeclaration javaTypeDeclaration;

	// ********** constructors **********

	/**
	 * this constructor is called when the mapping is read from an XML file
	 */
	AbstractJDBCTypeToJavaTypeDeclarationMapping(JDBCTypeRepository repository, Node node) throws CorruptXMLException {
		super(repository);
		this.read(node);
	}

	/**
	 * this constructor is called when the user (or a test case)
	 * creates a new mapping (which shouldn't happen very often,
	 * since all the typical mappings have already been built...)
	 */
	AbstractJDBCTypeToJavaTypeDeclarationMapping(JDBCTypeRepository repository, JDBCType jdbcType, String javaClassName, int arrayDepth) {
		super(repository);
		if (jdbcType == null) {
			throw new NullPointerException();
		}
		this.jdbcType = jdbcType;
		this.javaTypeDeclaration = new JavaTypeDeclaration(this, javaClassName, arrayDepth);
	}


	// ********** accessors **********

	private JDBCTypeRepository getJDBCTypeRepository() {
		return  (JDBCTypeRepository) this.getParent();
	}

	public JDBCType getJDBCType() {
		return this.jdbcType;
	}

	public JavaTypeDeclaration getJavaTypeDeclaration() {
		return this.javaTypeDeclaration;
	}


	// ********** queries **********

	private JDBCType jdbcTypeNamed(String jdbcTypeName) {
		return this.getJDBCTypeRepository().jdbcTypeNamed(jdbcTypeName);
	}


	// ********** i/o **********

	private void read(Node node) throws CorruptXMLException {
		try {
			this.jdbcType = this.jdbcTypeNamed(XMLTools.getChildTextContent(node, "jdbc-type", null));
		} catch (IllegalArgumentException ex) {
			throw new CorruptXMLException(ex);
		}
		this.javaTypeDeclaration = new JavaTypeDeclaration(this, XMLTools.getChild(node, "java-type-declaration"));
	}

	/**
	 * Subclasses decide the order the child nodes are written.
	 */
	abstract void write(Node node);

	void writeJDBCType(Node node) {
		XMLTools.addSimpleTextNode(node, "jdbc-type", this.jdbcType.getName());
	}

	void writeJavaTypeDeclaration(Node node) {
		this.javaTypeDeclaration.write(node.appendChild(node.getOwnerDocument().createElement("java-type-declaration")));
	}

	// ********** printing and displaying **********

	public abstract void displayStringOn(StringBuffer sb);

	/**
	 * @see org.eclipse.persistence.tools.workbench.utility.Node#displayString()
	 */
	@Override
	public String displayString() {
		StringBuffer sb = new StringBuffer();
		this.displayStringOn(sb);
		return sb.toString();
	}

	/**
	 * @see org.eclipse.persistence.tools.utility.model.workbench.utility.AbstractModel#toString(java.lang.StringBuffer)
	 */
	public void toString(StringBuffer sb) {
		this.displayStringOn(sb);
	}

}
