/*******************************************************************************
 * Copyright (c) 2012, 2015 Ericsson
 * Copyright (c) 2010, 2011 École Polytechnique de Montréal
 * Copyright (c) 2010, 2011 Alexandre Montplaisir <alexandre.montplaisir@gmail.com>
 *
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 *******************************************************************************/

package org.eclipse.tracecompass.tmf.core.statesystem;

import org.eclipse.jdt.annotation.Nullable;
import org.eclipse.tracecompass.statesystem.core.ITmfStateSystem;
import org.eclipse.tracecompass.statesystem.core.ITmfStateSystemBuilder;
import org.eclipse.tracecompass.tmf.core.event.ITmfEvent;
import org.eclipse.tracecompass.tmf.core.trace.ITmfTrace;

/**
 * This is the interface used to define the "state change input", which is the
 * main type of input that goes in the state system.
 *
 * Usually a state change input, also called "state provider" is the piece of
 * the pipeline which converts trace events to state changes.
 *
 * @author Alexandre Montplaisir
 */
public interface ITmfStateProvider {

    /**
     * Different future event type on the state system
     *
     * @author Katherine Nadeau
     * @since 5.0
     */
    public enum FutureEventType {
        /**
         * Modify attribute event
         */
        MODIFICATION,
        /**
         * push value to a stack
         */
        PUSH,
        /**
         * Pop from a stack
         */
        POP;
    }

    /**
     * Event handler plugins should provide a version number. This is used to
     * determine if a potential existing file can be re-opened later (if the
     * versions in the file and in the viewer match), or if the file should be
     * rebuilt from scratch (if the versions don't match).
     *
     * @return The version number of the input plugin
     */
    int getVersion();

    /**
     * Get the trace with which this state input plugin is associated.
     *
     * @return The associated trace
     */
    ITmfTrace getTrace();

    /**
     * Return the start time of this "state change input", which is normally the
     * start time of the originating trace (or it can be the time of the first
     * state-changing event).
     *
     * @return The start time
     */
    long getStartTime();

    /**
     * Return the last time at which it is safe to query the state system under
     * construction, ie not the current end time of the underlying state system,
     * but the time just before the latest event that has been processed.
     *
     * @return The last timestamp at which it is safe to query the state system
     *         underneath
     * @since 2.0
     */
    long getLatestSafeTime();

    /**
     * Assign the target state system where this SCI will insert its state
     * changes. Because of dependencies issues, this can normally not be done at
     * the constructor.
     *
     * This needs to be called before .run()!
     *
     * @param ssb
     *            Target state system for the state changes generated by this
     *            input plugin
     */
    void assignTargetStateSystem(ITmfStateSystemBuilder ssb);

    /**
     * Return the currently assigned target state system.
     *
     * @return Reference to the currently assigned state system, or null if no
     *         SS is assigned yet
     */
    @Nullable ITmfStateSystem getAssignedStateSystem();

    /**
     * Send an event to this input plugin for processing. The implementation
     * should check the contents, and call the state-modifying methods of its
     * IStateSystemBuilder object accordingly.
     *
     * @param event
     *            The event (which should be safe to cast to the
     *            expectedEventType) that has to be processed.
     */
    void processEvent(ITmfEvent event);

    /**
     * Provide a non-initialized copy of this state input plugin. You will need
     * to call {@link #assignTargetStateSystem} on it to assign its target.
     *
     * @return A new state change input object, of the same type, but without an
     *         assigned target state system
     */
    ITmfStateProvider getNewInstance();

    /**
     * Indicate to the state history building process that we are done (for now),
     * and that it should close its current history.
     */
    void dispose();

    /**
     * Callback when trace has been fully read
     * @since 2.1
     */
    default void done() {
        // do nothing
    }

    /**
     * Makes the state provider fail with a cause
     *
     * @param cause The cause of the failure
     * @since 3.0
     */
    default void fail(Throwable cause) {
        // Do nothing by default.
    }

    /**
     * Get the cause of failure, or <code>null</code> if there is no failure.
     *
     * @return The failure cause
     * @since 3.0
     */
    default @Nullable Throwable getFailureCause() {
        return null;
    }

    /**
     * Add future state. The value of the state will be changed at the requested
     * time when it is safe to do so, ie when the analysis has reached this
     * timestamp.
     *
     * @param time
     *            the time of the state
     * @param futureValue
     *            the value of the state
     * @param attribute
     *            the quark to set to initial state
     * @since 4.1
     */
    default void addFutureEvent(long time, @Nullable Object futureValue, int attribute) {
        addFutureEvent(time, futureValue, attribute, FutureEventType.MODIFICATION);
    }

    /**
     * Add future state. The value of the state will be changed at the requested
     * time when it is safe to do so, ie when the analysis has reached this
     * timestamp.
     *
     * @param time
     *            the time of the state
     * @param futureValue
     *            the value of the state
     * @param attribute
     *            the quark to set to initial state
     * @param type
     *            the type of event to apply to the state system
     * @since 5.0
     */
    default void addFutureEvent(long time, @Nullable Object futureValue, int attribute, FutureEventType type) {
        // Do nothing by default
    }
}
