/**
 * Copyright (c) 2016 TypeFox GmbH (http://www.typefox.io) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.elk.graph.text.ui.contentassist;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import org.eclipse.elk.core.data.ILayoutMetaData;
import org.eclipse.elk.core.data.LayoutAlgorithmData;
import org.eclipse.elk.core.data.LayoutMetaDataService;
import org.eclipse.elk.core.data.LayoutOptionData;
import org.eclipse.elk.core.options.CoreOptions;
import org.eclipse.elk.graph.ElkEdge;
import org.eclipse.elk.graph.ElkGraphElement;
import org.eclipse.elk.graph.ElkLabel;
import org.eclipse.elk.graph.ElkNode;
import org.eclipse.elk.graph.ElkPort;
import org.eclipse.elk.graph.impl.ElkPropertyToValueMapEntryImpl;
import org.eclipse.elk.graph.properties.IProperty;
import org.eclipse.elk.graph.text.services.ElkGraphGrammarAccess;
import org.eclipse.elk.graph.text.ui.contentassist.AbstractElkGraphProposalProvider;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.swt.graphics.Image;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.Assignment;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.Keyword;
import org.eclipse.xtext.RuleCall;
import org.eclipse.xtext.ui.IImageHelper;
import org.eclipse.xtext.ui.editor.contentassist.ContentAssistContext;
import org.eclipse.xtext.ui.editor.contentassist.ICompletionProposalAcceptor;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Special content assist proposals for the ELK Graph language.
 */
@SuppressWarnings("all")
public class ElkGraphProposalProvider extends AbstractElkGraphProposalProvider {
  private final static Set<String> DISABLED_KEYWORDS = Collections.<String>unmodifiableSet(CollectionLiterals.<String>newHashSet("}", "]"));
  
  @Inject
  private ElkGraphGrammarAccess grammar;
  
  @Inject
  private IImageHelper imageHelper;
  
  @Override
  public void completeKeyword(final Keyword keyword, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    if (((!ElkGraphProposalProvider.DISABLED_KEYWORDS.contains(keyword.getValue())) && (!Objects.equal(keyword.getValue(), context.getPrefix())))) {
      super.completeKeyword(keyword, context, acceptor);
    }
  }
  
  @Override
  protected boolean doCreateStringProposals() {
    return false;
  }
  
  @Override
  public void complete_PropertyKey(final EObject model, final RuleCall ruleCall, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    boolean _matched = false;
    if (model instanceof ElkNode) {
      _matched=true;
      if (((((ElkNode)model).getParent() == null) || (!((ElkNode)model).getChildren().isEmpty()))) {
        LayoutAlgorithmData _algorithm = this.getAlgorithm(((ElkGraphElement)model));
        this.proposeProperties(((ElkGraphElement)model), _algorithm, LayoutOptionData.Target.PARENTS, context, acceptor);
      }
      ElkNode _parent = ((ElkNode)model).getParent();
      boolean _tripleNotEquals = (_parent != null);
      if (_tripleNotEquals) {
        ElkNode _parent_1 = ((ElkNode)model).getParent();
        LayoutAlgorithmData _algorithm_1 = this.getAlgorithm(_parent_1);
        this.proposeProperties(((ElkGraphElement)model), _algorithm_1, LayoutOptionData.Target.NODES, context, acceptor);
      }
    }
    if (!_matched) {
      if (model instanceof ElkEdge) {
        _matched=true;
        LayoutAlgorithmData _algorithm = this.getAlgorithm(((ElkGraphElement)model));
        this.proposeProperties(((ElkGraphElement)model), _algorithm, LayoutOptionData.Target.EDGES, context, acceptor);
      }
    }
    if (!_matched) {
      if (model instanceof ElkPort) {
        _matched=true;
        LayoutAlgorithmData _algorithm = this.getAlgorithm(((ElkGraphElement)model));
        this.proposeProperties(((ElkGraphElement)model), _algorithm, LayoutOptionData.Target.PORTS, context, acceptor);
      }
    }
    if (!_matched) {
      if (model instanceof ElkLabel) {
        _matched=true;
        LayoutAlgorithmData _algorithm = this.getAlgorithm(((ElkGraphElement)model));
        this.proposeProperties(((ElkGraphElement)model), _algorithm, LayoutOptionData.Target.LABELS, context, acceptor);
      }
    }
  }
  
  private LayoutAlgorithmData getAlgorithm(final ElkGraphElement element) {
    final ElkNode node = EcoreUtil2.<ElkNode>getContainerOfType(element, ElkNode.class);
    if ((node != null)) {
      final String algorithmId = node.<String>getProperty(CoreOptions.ALGORITHM);
      boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(algorithmId);
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        LayoutMetaDataService _instance = LayoutMetaDataService.getInstance();
        return _instance.getAlgorithmDataBySuffix(algorithmId);
      }
    }
    return null;
  }
  
  private void proposeProperties(final ElkGraphElement element, final LayoutAlgorithmData algorithmData, final LayoutOptionData.Target targetType, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    final LayoutMetaDataService metaDataService = LayoutMetaDataService.getInstance();
    Collection<LayoutOptionData> _optionData = metaDataService.getOptionData();
    final Function1<LayoutOptionData, Boolean> _function = new Function1<LayoutOptionData, Boolean>() {
      @Override
      public Boolean apply(final LayoutOptionData o) {
        return Boolean.valueOf(((targetType == null) || o.getTargets().contains(targetType)));
      }
    };
    Iterable<LayoutOptionData> _filter = IterableExtensions.<LayoutOptionData>filter(_optionData, _function);
    final Function1<LayoutOptionData, Boolean> _function_1 = new Function1<LayoutOptionData, Boolean>() {
      @Override
      public Boolean apply(final LayoutOptionData o) {
        return Boolean.valueOf((((algorithmData == null) || algorithmData.knowsOption(o)) || Objects.equal(CoreOptions.ALGORITHM, o)));
      }
    };
    Iterable<LayoutOptionData> _filter_1 = IterableExtensions.<LayoutOptionData>filter(_filter, _function_1);
    final Function1<LayoutOptionData, Boolean> _function_2 = new Function1<LayoutOptionData, Boolean>() {
      @Override
      public Boolean apply(final LayoutOptionData o) {
        return Boolean.valueOf(((element == null) || (!element.getProperties().map().containsKey(o))));
      }
    };
    final Iterable<LayoutOptionData> filteredOptions = IterableExtensions.<LayoutOptionData>filter(_filter_1, _function_2);
    for (final LayoutOptionData option : filteredOptions) {
      {
        String _id = option.getId();
        final String[] split = _id.split("\\.");
        String suffix = null;
        boolean foundMatch = false;
        int _length = split.length;
        int i = (_length - 1);
        while (((i >= 0) && (!foundMatch))) {
          {
            if ((suffix == null)) {
              int _minusMinus = i--;
              String _get = split[_minusMinus];
              suffix = _get;
            } else {
              int _minusMinus_1 = i--;
              String _get_1 = split[_minusMinus_1];
              String _plus = (_get_1 + ".");
              String _plus_1 = (_plus + suffix);
              suffix = _plus_1;
            }
            if (((metaDataService.getOptionDataBySuffix(suffix) != null) && suffix.startsWith(context.getPrefix()))) {
              foundMatch = true;
            }
          }
        }
        StyledString _displayString = this.getDisplayString(option, suffix);
        Image _image = this.getImage(option, null);
        final ICompletionProposal proposal = this.createCompletionProposal(suffix, _displayString, _image, context);
        acceptor.accept(proposal);
      }
    }
  }
  
  @Override
  public void completeProperty_Value(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    if ((model instanceof ElkPropertyToValueMapEntryImpl)) {
      final IProperty<?> property = ((ElkPropertyToValueMapEntryImpl)model).getKey();
      if ((property instanceof LayoutOptionData)) {
        String[] _choices = ((LayoutOptionData)property).getChoices();
        for (final String choice : _choices) {
          {
            Image _image = this.getImage(((LayoutOptionData)property), choice);
            final ICompletionProposal proposal = this.createCompletionProposal(choice, choice, _image, context);
            acceptor.accept(proposal);
          }
        }
        boolean _equals = Objects.equal(CoreOptions.ALGORITHM, property);
        if (_equals) {
          this.proposeAlgorithms(context, acceptor);
        }
      }
    }
  }
  
  private void proposeAlgorithms(final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    final LayoutMetaDataService metaDataService = LayoutMetaDataService.getInstance();
    Collection<LayoutAlgorithmData> _algorithmData = metaDataService.getAlgorithmData();
    for (final LayoutAlgorithmData algorithm : _algorithmData) {
      {
        String _id = algorithm.getId();
        final String[] split = _id.split("\\.");
        String suffix = null;
        boolean foundMatch = false;
        int _length = split.length;
        int i = (_length - 1);
        while (((i >= 0) && (!foundMatch))) {
          {
            if ((suffix == null)) {
              int _minusMinus = i--;
              String _get = split[_minusMinus];
              suffix = _get;
            } else {
              int _minusMinus_1 = i--;
              String _get_1 = split[_minusMinus_1];
              String _plus = (_get_1 + ".");
              String _plus_1 = (_plus + suffix);
              suffix = _plus_1;
            }
            if (((metaDataService.getAlgorithmDataBySuffix(suffix) != null) && suffix.startsWith(context.getPrefix()))) {
              foundMatch = true;
            }
          }
        }
        StyledString _displayString = this.getDisplayString(algorithm, suffix);
        final ICompletionProposal proposal = this.createCompletionProposal(suffix, _displayString, null, context);
        acceptor.accept(proposal);
      }
    }
  }
  
  private StyledString getDisplayString(final ILayoutMetaData data, final String suffix) {
    StyledString _styledString = new StyledString(suffix);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(" ");
    _builder.append("–", " ");
    _builder.append(" ");
    String _name = data.getName();
    _builder.append(_name, " ");
    _builder.append(" (");
    String _id = data.getId();
    _builder.append(_id, " ");
    _builder.append(")");
    StyledString _styledString_1 = new StyledString(_builder.toString(), StyledString.QUALIFIER_STYLER);
    return this.operator_plus(_styledString, _styledString_1);
  }
  
  @Override
  protected Image getImage(final EObject eObject) {
    if ((eObject instanceof Keyword)) {
      String _switchResult = null;
      boolean _matched = false;
      ElkGraphGrammarAccess.RootNodeElements _rootNodeAccess = this.grammar.getRootNodeAccess();
      Keyword _graphKeyword_1_0 = _rootNodeAccess.getGraphKeyword_1_0();
      if (Objects.equal(eObject, _graphKeyword_1_0)) {
        _matched=true;
        _switchResult = "elkgraph";
      }
      if (!_matched) {
        ElkGraphGrammarAccess.ElkNodeElements _elkNodeAccess = this.grammar.getElkNodeAccess();
        Keyword _nodeKeyword_0 = _elkNodeAccess.getNodeKeyword_0();
        if (Objects.equal(eObject, _nodeKeyword_0)) {
          _matched=true;
          _switchResult = "elknode";
        }
      }
      if (!_matched) {
        ElkGraphGrammarAccess.ElkEdgeElements _elkEdgeAccess = this.grammar.getElkEdgeAccess();
        Keyword _edgeKeyword_0 = _elkEdgeAccess.getEdgeKeyword_0();
        if (Objects.equal(eObject, _edgeKeyword_0)) {
          _matched=true;
          _switchResult = "elkedge";
        }
      }
      if (!_matched) {
        ElkGraphGrammarAccess.ElkPortElements _elkPortAccess = this.grammar.getElkPortAccess();
        Keyword _portKeyword_0 = _elkPortAccess.getPortKeyword_0();
        if (Objects.equal(eObject, _portKeyword_0)) {
          _matched=true;
          _switchResult = "elkport";
        }
      }
      if (!_matched) {
        ElkGraphGrammarAccess.ElkLabelElements _elkLabelAccess = this.grammar.getElkLabelAccess();
        Keyword _labelKeyword_0 = _elkLabelAccess.getLabelKeyword_0();
        if (Objects.equal(eObject, _labelKeyword_0)) {
          _matched=true;
          _switchResult = "elklabel";
        }
      }
      final String key = _switchResult;
      if ((key != null)) {
        return this.imageHelper.getImage((key + ".gif"));
      }
    }
    return super.getImage(eObject);
  }
  
  private Image getImage(final LayoutOptionData option, final String value) {
    String _switchResult = null;
    LayoutOptionData.Type _type = option.getType();
    if (_type != null) {
      switch (_type) {
        case BOOLEAN:
          String _xifexpression = null;
          boolean _equals = Objects.equal(value, "false");
          if (_equals) {
            _xifexpression = "prop_false";
          } else {
            _xifexpression = "prop_true";
          }
          _switchResult = _xifexpression;
          break;
        case INT:
          _switchResult = "prop_int";
          break;
        case DOUBLE:
          _switchResult = "prop_double";
          break;
        case ENUM:
        case ENUMSET:
          _switchResult = "prop_choice";
          break;
        default:
          _switchResult = "prop_text";
          break;
      }
    } else {
      _switchResult = "prop_text";
    }
    final String key = _switchResult;
    return this.imageHelper.getImage((key + ".gif"));
  }
  
  private StyledString operator_plus(final StyledString s1, final StyledString s2) {
    return s1.append(s2);
  }
}
